
#define _CRT_SECURE_NO_WARNINGS
#include <atomic>
#include <mutex>
#include <vector>
#include <string>
#include <thread>
#include <chrono>
#include <iostream>
#include <deque>
#include <algorithm>

#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <Windows.h>
#include <gl/GL.h>

typedef BOOL (WINAPI * PFNWGLJOINSWAPGROUPNVPROC) (HDC hDC, GLuint group);
typedef BOOL (WINAPI * PFNWGLBINDSWAPBARRIERNVPROC) (GLuint group, GLuint barrier);
typedef BOOL (WINAPI * PFNWGLSWAPINTERVALEXTPROC) (int interval);

std::mutex g_mutex;
std::atomic<bool> g_start;

LRESULT CALLBACK wnd_proc(HWND hwnd, UINT umsg, WPARAM wparam, LPARAM lparam) {
  switch (umsg) {
    case WM_CLOSE:
      PostQuitMessage(0);
      break; 

    case WM_KEYDOWN:
      if (wparam == 27)
        PostQuitMessage(0);
      break; 

    case WM_ERASEBKGND:
      return 1;

    case WM_PAINT:
      ValidateRect(hwnd, nullptr);
      return 0;
  }
  return DefWindowProcW(hwnd, umsg, wparam, lparam);
}

void register_window_class() {
  WNDCLASSW wnd_class{ };
  wnd_class.lpszClassName = L"GL_WINDOW_CLASS";
  wnd_class.lpfnWndProc = (WNDPROC)wnd_proc;
  wnd_class.hInstance = GetModuleHandleW(0);
  wnd_class.style = CS_OWNDC;
  wnd_class.hIcon = LoadIcon(NULL, IDI_APPLICATION);
  RegisterClassW(&wnd_class);
}

void draw_tearing_indicator(double time_s, double period_time_s,
    int line_width, int w, int h) {

  double pos = 2.0 * fmod(time_s / period_time_s, 1) - 1.0;
  int left = static_cast<int>(fabs(pos) * (w - line_width+1));
  int top = static_cast<int>(fabs(pos) * (h - line_width+1));
  glEnable(GL_SCISSOR_TEST);
  glClearColor(0, 1, 0, 0);
  glScissor(left, 0, line_width, h);
  glClear(GL_COLOR_BUFFER_BIT);
  glScissor(0, top, w, line_width);
  glClear(GL_COLOR_BUFFER_BIT);

  glScissor(0, 0, 2, w);
  glClear(GL_COLOR_BUFFER_BIT);
  glScissor(w - 2, 0, 2, h);
  glClear(GL_COLOR_BUFFER_BIT);
  glScissor(0, 0, w, 2);
  glClear(GL_COLOR_BUFFER_BIT);
  glScissor(0, h - 2, w, 2);
  glClear(GL_COLOR_BUFFER_BIT);

  glDisable(GL_SCISSOR_TEST);
}

bool message_loop() {
  MSG msg;
  while (PeekMessageW(&msg, NULL, 0, 0, PM_REMOVE)) {
    if (msg.message == WM_QUIT) {
      return false;
    }
    else {
      TranslateMessage(&msg);
      DispatchMessageW(&msg);
    }
  }
  return true;
}

class Window {
public:
  Window(int x, int y, int width, int height, char swap_behavior) 
    : m_x(x), m_y(y), m_width(width), m_height(height) {
    m_thread = std::thread(&Window::thread_func, this, swap_behavior);
  }

  ~Window() {
    if (m_thread.joinable())
      m_thread.join();
  }

private:
  void thread_func(char swap_behavior) {
    std::unique_lock<std::mutex> lock{ g_mutex };

    m_window_handle = CreateWindowExW(WS_EX_APPWINDOW, L"GL_WINDOW_CLASS",  
      L"Swap Group Test", WS_POPUP, m_x, m_y, m_width, m_height, 
      NULL, NULL, GetModuleHandle(0), NULL);
    m_device_context = GetDC(m_window_handle);

    // create OpenGL context
    PIXELFORMATDESCRIPTOR pfd{ };
    pfd.nSize = sizeof(pfd);
    pfd.nVersion = 1;
    pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
    int pixel_format = ChoosePixelFormat(m_device_context, &pfd);
    SetPixelFormat(m_device_context, pixel_format, &pfd);
    m_gl_context = wglCreateContext(m_device_context);

    wglMakeCurrent(m_device_context, m_gl_context);

    // join swap group
    if (swap_behavior == 'S') {
      if (auto wglJoinSwapGroupNV = (PFNWGLJOINSWAPGROUPNVPROC)wglGetProcAddress("wglJoinSwapGroupNV"))
        if (wglJoinSwapGroupNV(m_device_context, 1) != GL_FALSE)
          std::cout << "window at " << m_x << "," << m_y << " joined swap group" << std::endl;

      // bind swap barrier
      if (auto wglBindSwapBarrierNV = (PFNWGLBINDSWAPBARRIERNVPROC)wglGetProcAddress("wglBindSwapBarrierNV"))
        if (wglBindSwapBarrierNV(1, 1) != GL_FALSE)
          std::cout << "bound swap barrier" << std::endl;
    }
    else {
      const auto swap_interval = 
        (swap_behavior == 'V' ? 1 : 
         swap_behavior == 'T' ? -1 : 0);
      if (auto wglSwapIntervalEXT = (PFNWGLSWAPINTERVALEXTPROC)wglGetProcAddress("wglSwapIntervalEXT"))
        if (wglSwapIntervalEXT(swap_interval))
          std::cout << "window at " << m_x << "," << m_y << " set swap interval " << swap_interval << std::endl;
    }

    lock.unlock();

    while (!g_start.load())
      std::this_thread::sleep_for(std::chrono::milliseconds(1));

    ShowWindow(m_window_handle, SW_SHOW);

    auto last_swap_time = std::chrono::high_resolution_clock::now();
    double accumulator = 0;
    int sample_count = 0;
    double maximum = 0;

    while (message_loop()) {
      // redraw
      GLbitfield mask{ };
      glClearColor(0, 0, 0, 0);
      glClear(GL_COLOR_BUFFER_BIT);
      auto time = std::chrono::duration<double>(std::chrono::high_resolution_clock::now().time_since_epoch());
      draw_tearing_indicator(time.count(), 10, 10, m_width, m_height);

      // swap buffers
      SwapBuffers(m_device_context);

      // update statistics
      auto swap_time = std::chrono::high_resolution_clock::now();
      auto time_elapsed = std::chrono::duration<double>(swap_time - last_swap_time).count();
      accumulator += time_elapsed;
      maximum = std::max(maximum, time_elapsed);
      ++sample_count;
      if (sample_count == 100) {
        std::lock_guard<std::mutex> lock{ g_mutex };
        std::cout << "window at " << m_x << "," << m_y << 
          " refreshes at " << 1.0 / (accumulator / sample_count) << "Hz (min. " << 
            (1.0 / maximum) << "Hz)" << std::endl;
        accumulator = 0;
        sample_count = 0;
        maximum = 0;
      }
      last_swap_time = swap_time;
    }

    // shutdown
    wglMakeCurrent(0, 0);
    wglDeleteContext(m_gl_context);
    ReleaseDC(m_window_handle, m_device_context);
    DestroyWindow(m_window_handle);

    m_gl_context = 0;
    m_device_context = 0;
    m_window_handle = 0;
  }
  
  std::thread m_thread;
  int m_x, m_y, m_width, m_height;
  HWND m_window_handle{ };
  HDC m_device_context{ };
  HGLRC m_gl_context{ };
};

int APIENTRY WinMain(HINSTANCE, HINSTANCE, LPSTR, int) {
  if (__argc <= 5 || __argc % 5 != 1) {
    MessageBoxA(NULL, 
      "Please run from the command shell and provide \n"
      "the dimensions of each window to open (x, y, width, height) "
      "and the swap behavior (S: join swap group, V: wait for vsync, T: swap control tear, N: no sync).\n"
      "e.g. SwapGroupTest  0 0 1920 1200 S   1920 0 1920 1200 S", 
      "usage", MB_OK);
    return 1;
  }

  // redirect console output
  if (AttachConsole(ATTACH_PARENT_PROCESS)) {
    freopen("CONOUT$", "w", stdout);
    freopen("CONOUT$", "w", stderr);
  }
  std::cout << std::endl;

  register_window_class();  

  std::deque<Window> windows;
  for (int i = 1; i < __argc; i += 5) {
    windows.emplace_back(
      std::atoi(__argv[i + 0]),
      std::atoi(__argv[i + 1]),
      std::atoi(__argv[i + 2]),
      std::atoi(__argv[i + 3]),
      __argv[i + 4][0]);
  }

  std::this_thread::sleep_for(std::chrono::seconds(1));
  g_start.store(true);
  windows.clear();

  std::cout << "finished." << std::endl << std::endl;
  return 0;
}

